
pro read_spectra, FILENAME, PLOTLOS

;+
; NAME:
;       read_spectra.pro
;
; PURPOSE: Read Lyman-alpha forest spectra extracted from the Sherwood
;       simulation (Bolton et al. 2017, MNRAS, 464, 897)
;
; INPUTS:
;       FILENAME:  Location of the file containing the spectra (binary format)
;       PLOTLOS:  The line-of-sight to be plotted (value of 0 to 4999)
;
; OUTPUTS:
;       Plots transmitted flux against observed wavelength for
;       line-of-sight PLOTLOS
;
;       Returns the position of line-of-sight PLOTLOS within the
;       simulation box
;
;       Returns the effective optical depth (the average over all
;       lines-of-sight in the file) and HI photo-ionisation rate at
;       the output redshift
;
; EXAMPLE:
;       .r read_spectra
;       read_spectra,'./spectra_40_2048/tauH1_lya_z2.0.dat',3982
;
; MODIFICATION HISTORY:
;       Written by James Bolton, Sep 2017
;-

openr,1,FILENAME, ERROR=err
if(err ne 0) then begin
   print
   print,'Incorrect file location. Stopping...'
   print
   stop
endif

npix   = 0L
nlos   = 0L
readu,1,npix,nlos

if (PLOTLOS gt nlos-1) then begin
   print
   print,'PLOTLOS must be between 0 and ',strcompress(nlos-1,/remove_all),'. Stopping...'
   print
   close,1
   stop
endif

ztime   = 0.0 ; Redshift
omegaM  = 0.0 ; Mass density
omegaL  = 0.0 ; Vacuum energy density
omegab  = 0.0 ; Baryon density
hubble  = 0.0 ; Hubble constant [h=H0/100 km s^-1 Mpc^-1]
boxsize = 0.0 ; Box size [h^-1 ckpc]
readu,1,ztime,omegaM,omegaL,omegab,hubble,boxsize

print
print,'Reading spectra at z=',strcompress(ztime,/remove_all)

axis   = lonarr(nlos) ; Line-of-sight axis (x=1, y=2, z=3)
coord1 = fltarr(nlos) ; Line-of-sight plane, position 1 [h^-1 ckpc]
coord2 = fltarr(nlos) ; Line-of-sight plane, position 2 [h^-1 ckpc]
readu,1,axis,coord1,coord2

@get_los_position ; Returns location of sight-line in simulation box

pixels = fltarr(npix) ; Pixel positions [h^-1 ckpc]
readu,1,pixels

tau = fltarr(npix*nlos) ; HI Lyman-alpha optical depths
readu,1,tau
close,1

@get_uvb ; Returns HI photo-ionisation rate [s^-1]
@get_wavelength ; Calculates observed frame wavelength [Angstroms]

taueff = -1.0*alog(mean(exp(-tau))) ; effective optical depth
print,'The effective optical depth is ',strcompress(taueff,/remove_all)
print

;; Plot transmitted flux, F=exp(-tau)
window,0,xsize=1200,ysize=400
Device,Retain=2
!p.font=-1
plot,lambda,exp(-tau[PLOTLOS*npix :(PLOTLOS+1)*npix-1]),xstyle=1,ystyle=1,charsize=1.75,yrange=[-0.1,1.1],ytitle='Transmitted flux',xtitle='Wavelength [Angstroms]'
      
end
